/*==============================================================================
Replication package for the BPEA Spring 2024 paper:

"The Emergence of a Uniform Business Cycle in the United States: 
Evidence from New Claims-Based Unemployment Data"

Andrew Fieldhouse, David Munro, Christoffer Koch, and Sean Howard

Stata code for replicating Figure 5 (Local Projections-IV regressions)
==============================================================================*/

clear all

*Set Your Directory to Replication Package Folder:

* Andrew:
cd "/Users/afieldhouse/Dropbox/Research/Historical Claims Project/continuous_updates/Replication Package_BPEA_CBUR/Rep_Code/"

local dirdata "/Users/afieldhouse/Dropbox/Research/Historical Claims Project/continuous_updates/Replication Package_BPEA_CBUR/Rep_Data/"
local dircbur "/Users/afieldhouse/Dropbox/Research/Historical Claims Project/continuous_updates/Replication Package_BPEA_CBUR/CBUR Data/"
local dirfigs "/Users/afieldhouse/Dropbox/Research/Historical Claims Project/continuous_updates/Replication Package_BPEA_CBUR/Rep_Figures/"
local dirfigexp "/Users/afieldhouse/Dropbox/Research/Historical Claims Project/continuous_updates/Replication Package_BPEA_CBUR/Rep_Figures_Data_Export/"

***********************************************************
* Top Row of Figure 5: Full Sample: LP regressions with Bartik instrument ("rimix")
***********************************************************

*** Import LPIV dataset ***
u "`dirdata'LPIV_DATA.dta"

***Select full sample:
drop if year < 1950 | year > 2019

*Initialize matrix for IRFs
mat irfNFE = J(13,2,0)
mat irfPOP = J(13,2,0)
mat irfUR = J(13,2,0)

***** Estimate Impulse Responses for Fig 5a: Relative Employment *****
 
forvalues h = 0/11{

	quietly reg cNFE`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bNFE = _b[rimix]
	mat seNFE = _se[rimix]
	
	mat irfNFE[`h'+2,1] = bNFE 
	mat irfNFE[`h'+2,2] = seNFE
	
}
matlist irfNFE


***** Estimate Impulse Responses for Fig 5b: Relative Unemployment *****

forvalues h = 0/11{
	 
	quietly reg cUR`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)
	
	mat bUR = _b[rimix]
	mat seUR = _se[rimix]
	
	mat irfUR[`h'+2,1] = bUR
	mat irfUR[`h'+2,2] = seUR

}
matlist irfUR 

***** Estimate Impulse Responses for Fig 5c: Relative Population *****

forvalues h = 0/11{
	
	quietly reg cPOP`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bPOP = _b[rimix]
	mat sePOP = _se[rimix]
	
	mat irfPOP[`h'+2,1] = bPOP
	mat irfPOP[`h'+2,2] = sePOP
}
matlist irfPOP

**** Plot IRFs for Top Row of Fig 5 *****

gen long hor = _n 
	replace hor = . if hor > 13
	replace hor = hor-1
	la var hor "Years"

***** Plot Fig 5a: Relative Employment *****	
	
gen irfNFE = -irfNFE[_n,1]
	label var irfNFE "Employment Impulse Response"
gen uciNFE = irfNFE + 1.96*irfNFE[_n,2]
	label var uciNFE "Upper Confidence Band"
gen lciNFE = irfNFE - 1.96*irfNFE[_n,2]
	label var lciNFE "Lower Confidence Band"
	
twoway rarea lciNFE uciNFE hor, fc(gs12%60) lw(none) || line irfNFE hor, legend(off) title("Sample: 1950-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-2(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Emp_1950-2019.pdf", replace

export excel hor irfNFE uciNFE lciNFE using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5a_TopRow", replace) firstrow(varlabels)

***** Plot Fig 5b: Relative Unemployment *****	
	
gen irfUR = -irfUR[_n,1]
	label var irfUR "Unemployment Impulse Response"
gen uciUR = irfUR + 1.96*irfUR[_n,2]
	label var uciUR "Upper Confidence Band"
gen lciUR = irfUR - 1.96*irfUR[_n,2]
	label var lciUR "Lower Confidence Band"

twoway rarea lciUR uciUR hor, fc(gs12%60) lw(none) || line irfUR hor, legend(off) title("Sample: 1950-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-0.1(.1)0.2, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Unemp_1950-2019.pdf", replace

export excel hor irfUR uciUR lciUR using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5b_TopRow", replace) firstrow(varlabels)

***** Plot Fig 5c: Relative Population *****	
	
gen irfPOP = -irfPOP[_n,1]
	label var irfPOP "Population Impulse Response"
gen uciPOP = irfPOP + 1.96*irfPOP[_n,2]
	label var uciPOP "Upper Confidence Band"
gen lciPOP = irfPOP - 1.96*irfPOP[_n,2]
	label var lciPOP "Lower Confidence Band"
	
twoway rarea lciPOP uciPOP hor, fc(gs12%60) lw(none) || line irfPOP hor, legend(off) title("Sample: 1950-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-1.5(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Pop_1950-2019.pdf", replace

export excel hor irfPOP uciPOP lciPOP using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5c_TopRow", replace) firstrow(varlabels)

***********************************************************
* Middle Row of Figure 5: Early Subsample: LP regressions with Bartik instrument ("rimix")
***********************************************************

clear all

*** Import LPIV dataset ***
u "`dirdata'LPIV_DATA.dta"

***Select second half of sample:
drop if year < 1950 | year > 1985

*Initialize matrix for IRFs
mat irfNFE = J(13,2,0)
mat irfPOP = J(13,2,0)
mat irfUR = J(13,2,0)

***** Estimate Impulse Responses for Fig 5a: Relative Employment *****
 
forvalues h = 0/11{

	quietly reg cNFE`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bNFE = _b[rimix]
	mat seNFE = _se[rimix]
	
	mat irfNFE[`h'+2,1] = bNFE 
	mat irfNFE[`h'+2,2] = seNFE
	
}
matlist irfNFE

***** Estimate Impulse Responses for Fig 5b: Relative Unemployment *****

forvalues h = 0/11{
	 
	quietly reg cUR`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bUR = _b[rimix]
	mat seUR = _se[rimix]
	
	mat irfUR[`h'+2,1] = bUR
	mat irfUR[`h'+2,2] = seUR

}
matlist irfUR 

***** Estimate Impulse Responses for Fig 5c: Relative Population *****

forvalues h = 0/11{
	
	quietly reg cPOP`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bPOP = _b[rimix]
	mat sePOP = _se[rimix]
	
	mat irfPOP[`h'+2,1] = bPOP
	mat irfPOP[`h'+2,2] = sePOP
}
matlist irfPOP


**** Plot IRFs for Middle Row of Fig 5 *****

gen long hor = _n 
	replace hor = . if hor > 13
	replace hor = hor-1
	la var hor "Years"

***** Plot Fig 5a: Relative Employment  *****	
	
gen irfNFE = -irfNFE[_n,1]
	label var irfNFE "Employment Impulse Response"
gen uciNFE = irfNFE + 1.96*irfNFE[_n,2]
	label var uciNFE "Upper Confidence Band"
gen lciNFE = irfNFE - 1.96*irfNFE[_n,2]
	label var lciNFE "Lower Confidence Band"

twoway rarea lciNFE uciNFE hor, fc(gs12%60) lw(none) || line irfNFE hor, legend(off) title("Sample: 1950-1985", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-2(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Emp_1950-1985.pdf", replace

export excel hor irfNFE uciNFE lciNFE using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5a_MiddleRow", replace) firstrow(varlabels)

***** Plot Fig 5b: Relative Unemployment *****	
	
gen irfUR = -irfUR[_n,1]
	label var irfUR "Unemployment Impulse Response"
gen uciUR = irfUR + 1.96*irfUR[_n,2]
	label var uciUR "Upper Confidence Band"
gen lciUR = irfUR - 1.96*irfUR[_n,2]
	label var lciUR "Lower Confidence Band"

twoway rarea lciUR uciUR hor, fc(gs12%60) lw(none) || line irfUR hor, legend(off) title("Sample: 1950-1985", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-0.1(.1)0.2, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Unemp_1950-1985.pdf", replace

export excel hor irfUR uciUR lciUR using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5b_MiddleRow", replace) firstrow(varlabels)

***** Plot Fig 5c: Relative Employment Population *****	
	
gen irfPOP = -irfPOP[_n,1]
	label var irfPOP "Population Impulse Response"
gen uciPOP = irfPOP + 1.96*irfPOP[_n,2]
	label var uciPOP "Upper Confidence Band"
gen lciPOP = irfPOP - 1.96*irfPOP[_n,2]
	label var lciPOP "Lower Confidence Band"

twoway rarea lciPOP uciPOP hor, fc(gs12%60) lw(none) || line irfPOP hor, legend(off) title("Sample: 1950-1985", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-1.5(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Pop_1950-1985.pdf", replace

export excel hor irfPOP uciPOP lciPOP using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5c_MiddleRow", replace) firstrow(varlabels)

***********************************************************
* Bottom Row of Figure 5: Recent Subsample: LP regressions with Bartik instrument ("rimix")
***********************************************************

clear all

*** Import LPIV dataset ***
u "`dirdata'LPIV_DATA.dta"

*u "`dirdata'Bartik_IRF_Figs.dta"

***Select second half of sample:
drop if year < 1986 | year > 2019


*Initialize matrix for IRFs
mat irfNFE = J(13,2,0)
mat irfPOP = J(13,2,0)
mat irfUR = J(13,2,0)

***** Estimate Impulse Responses for Fig 5a: Relative Employment *****
 
forvalues h = 0/11{

	quietly reg cNFE`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bNFE = _b[rimix]
	mat seNFE = _se[rimix]
	
	mat irfNFE[`h'+2,1] = bNFE 
	mat irfNFE[`h'+2,2] = seNFE
	
}
matlist irfNFE

***** Estimate Impulse Responses for Fig 5b: Relative Unemployment *****

forvalues h = 0/11{
	 
	quietly reg cUR`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bUR = _b[rimix]
	mat seUR = _se[rimix]
	
	mat irfUR[`h'+2,1] = bUR
	mat irfUR[`h'+2,2] = seUR

}
matlist irfUR 

***** Estimate Impulse Responses for Fig 5c: Relative Population *****

forvalues h = 0/11{
	
	quietly reg cPOP`h' rimix reldNFE_L1 reldCBUR_L1 reldPOP_L1 reldNFE_L2 reldCBUR_L2 reldPOP_L2 i.fips i.year, vce(cl fips)

	mat bPOP = _b[rimix]
	mat sePOP = _se[rimix]
	
	mat irfPOP[`h'+2,1] = bPOP
	mat irfPOP[`h'+2,2] = sePOP
}
matlist irfPOP

**** Plot IRFs for Bottom Row of Fig 5 *****

gen long hor = _n 
	replace hor = . if hor > 13
	replace hor = hor-1
	la var hor "Years"

***** Plot Fig 5a: Relative Employment *****	
	
gen irfNFE = -irfNFE[_n,1]
	label var irfNFE "Employment Impulse Response"
gen uciNFE = irfNFE + 1.96*irfNFE[_n,2]
	label var uciNFE "Upper Confidence Band"
gen lciNFE = irfNFE - 1.96*irfNFE[_n,2]
	label var lciNFE "Lower Confidence Band"

twoway rarea lciNFE uciNFE hor, fc(gs12%60) lw(none) || line irfNFE hor, legend(off) title("Sample: 1986-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-2(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Emp_1986-2019.pdf", replace

export excel hor irfNFE uciNFE lciNFE using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5a_BottomRow", replace) firstrow(varlabels)


***** Plot Fig 5b: Relative Unemployment *****	
	
gen irfUR = -irfUR[_n,1]
	label var irfUR "Unemployment Impulse Response"
gen uciUR = irfUR + 1.96*irfUR[_n,2]
	label var uciUR "Upper Confidence Band"
gen lciUR = irfUR - 1.96*irfUR[_n,2]
	label var lciUR "Lower Confidence Band"

twoway rarea lciUR uciUR hor, fc(gs12%60) lw(none) || line irfUR hor, legend(off) title("Sample: 1986-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-0.1(.1)0.2, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Unemp_1986-2019.pdf", replace

export excel hor irfUR uciUR lciUR using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5b_BottomRow", replace) firstrow(varlabels)

***** Plot Fig 5c: Relative Population *****	
	
gen irfPOP = -irfPOP[_n,1]
	label var irfPOP "Population Impulse Response"
gen uciPOP = irfPOP + 1.96*irfPOP[_n,2]
	label var uciPOP "Upper Confidence Band"
gen lciPOP = irfPOP - 1.96*irfPOP[_n,2]
	label var lciPOP "Lower Confidence Band"

twoway rarea lciPOP uciPOP hor, fc(gs12%60) lw(none) || line irfPOP hor, legend(off) title("Sample: 1986-2019", span) ytitle("Percent") xtitle("Years")lw(thick) lc(navy)  xlab(0(1)12) yline(0, lw(thin) lc(gs12)) ylab(-1.5(.5)0.5, format(%4.1f)) ymt(##5)  xsize(4.8) ysize(3)

graph export "`dirfigs'/Fig5_Pop_1986-2019.pdf", replace

export excel hor irfPOP uciPOP lciPOP using "`dirfigexp'Fig5_Export.xlsx", sheet("Fig5c_BottomRow", replace) firstrow(varlabels)


// END OF FILE //

